<?php
class GateCurls {
    public $booking_availability    = "dc_tvg/gate_tvg/booking_availability";
    public $booking_submit          = "dc_tvg/gate_tvg/booking_submit";
    public $booking_cancel          = "dc_tvg/gate_tvg/booking_cancel";
}

class BaseTvgClass {
    public $properties;
    public function __construct($construct_data){
        $this->properties = new stdClass();
        if (isset($construct_data['props'])){
            foreach ($construct_data['props'] as $prop_name => $prop_data) {
                switch ($prop_data['type']) {
                    case "integer":
                        $val = 0;
                        break;
                    case "string":
                        $val = '';
                        break;
                    case "array":
                        $val = [];
                        break;
                    case "object":
                        $val = new $prop_data['class'];
                        //$val = $val->properties;
                        break;
                    default:
                        $val = '';
                }
                $this->properties->$prop_name = $val;
            }
        }
    }

    public function print_props() {
        echo "<pre>";
        print_r( json_decode(json_encode($this->properties, true)));
    }
}

class Request extends BaseTvgClass{
    public function __construct($construct_data){
        parent::__construct($construct_data);
    }
}

class Response extends BaseTvgClass{
    public function __construct($construct_data){
        parent::__construct($construct_data);
    }

    private function validate(&$obj_or_arr) {
        foreach ( $obj_or_arr as $p => $v ) {
            if (!in_array($p, $this->required)){
                if (is_array($v) || is_object($v)){
                    $this->validate($v);
                    if (!count((array)$v)){
                        unset($obj_or_arr->$p);
                    }
                }elseif (is_null($v) || empty($v)) {
                    unset($obj_or_arr->$p);
                }
            }
        }
    }
//     public function get_request(){
//         return json_decode($this->request);
//     }
//     public function makelog() {
//         $data = array (
//             'rawpost' => json_encode($this, true)
//         );
//         $this->db->insert ( 'dc_booking.ta_http_response', $data );
//     }
//     public function get_response() {
//         return json_encode($this, true);
//     }

    public function send($debug_mode = FALSE) {
        if ($debug_mode){
            echo "<pre>";
            print_r($this);
            die("DIED. Controller class Ta_ib_ep {static debug_mode = TRUE}");
        }else{
            self::validate($this->properties);
            header ( 'Content-Type: application/json' );
            echo json_encode($this->properties, true);
        }
    }
}

class BookingAvailabilityRequest extends Request{
    private $title = "Booking Availability Request";
    private $description = "Schema of booking availability request object";
    private $required = [
        "api_version",
        "hotel",
        "start_date",
        "end_date",
        "party",
        "lang",
        "currency",
        "user_country"
    ];
    private $_properties = [
        "api_version" => [
            "description"=> "Version of the API used.",
            "type"=> "integer",
            "example"=> 1
        ],
        "hotel" => [
            "type" => "object",
            "class" => "Hotel"
        ],
        "start_date" => [
            "description"=> "The user's check-in date in YYYY-MM-DD",
            "type"=> "string",
            "format"=> "date",
            "example"=> "2019-01-01"
        ],
        "end_date"=> [
            "description"=> "The user's check-out date in YYYY-MM-DD",
            "type"=> "string",
            "format"=> "date",
            "example"=> "2019-01-02"
        ],
        "party"=> [
            "description"=> "Array of JSON objects inlcuding number of adults and children requested.\nEach party object represents the occupation for a separate room.\nFor example, \"party\"=>[[\"adults\"=> 3]] is a request for one room that accommodates three adults,\nor \"party\"=>[[\"adults\"=> 2],[\"adults\"=> 2, \"children\"=> [11,8]]] is a request for one room for two adults\nand one room for two adults and two children aged 11 and 8.",
            "type"=> "array",
            "items"=> [
                "Party Object"
            ],
            "minItems"=> 1
        ],
        "lang"=> [
            "description"=> "Language code (see Supported Languages)[http=>//developer.trivago.com/expressbooking/supported-languages.html], which identifies the corresponding language and POS.",
            "type"=> "string",
            "example"=> "en_US"
        ],
        "currency"=> [
            "description"=> "ISO 4217 currency code. All prices, taxes, fees, and discount amounts should be returned in this requested currency.",
            "type"=> "string",
            "example"=> "USD"
        ],
        "user_country"=> [
            "description"=> "ISO 3166-1 alpha-2 country code corresponding to the user making this request.",
            "type"=> "string",
            "example"=> "US"
        ]
    ];

    public function __construct($request=''){
        parent::__construct(['props'=>$this->_properties]);
        $request = preg_replace("/\r|\n|\s/", "", $request);
        $this->origin = $request;
        parse_str($request, $request);
        foreach ($request as $prop_name => $strvalue) {
            if (strpbrk($request[$prop_name], '[]{}')) {
                $request[$prop_name] = json_decode($strvalue);
            }
            if (isset($this->properties->$prop_name)){
                $this->properties->$prop_name = $request[$prop_name];
            }
        }
        //if validate
        $this->_set_adapted_hotel_properties();
    }

    private function _set_adapted_hotel_properties() {
        $this->properties->int_datein = round (strtotime($this->properties->start_date)/86400);
        $this->properties->int_dateout = round (strtotime($this->properties->end_date)/86400);
        $this->properties->rooms = count($this->properties->party);
        $this->properties->persons = $this->get_requested_guests_count();
        $this->properties->prop_id = $this->get_hotel_property_id_from_reference_code($this->properties->hotel->partner_reference);
    }

    private function get_requested_guests_count (){
        $max_guests_count = 1;
        foreach ($this->properties->party as $room_party) {
            $children_count = (isset($room_party->children)) ? count($room_party->children) : 0;
            $room_guests_count = $room_party->adults + $children_count;
            if ($room_guests_count > $max_guests_count){ $max_guests_count = $room_guests_count;}
        }
        return $max_guests_count;
    }

    private function get_hotel_property_id_from_reference_code($reference_code) {
        $parsed_ids_arr = explode('_', $reference_code);
        return $parsed_ids_arr[1];
    }

}

class BookingAvailabilityResponce extends Response{
    private $title= "Booking Availability Ok";
    private $description = "Schema Booking availability when response is OK";
    protected $required = [
        "api_version",
        "hotel_id",
        "start_date",
        "end_date",
        "party",
        "lang",
        "user_country",
        "room_types_array",
        "customer_support"
    ];
    private $_properties = [
        "api_version"=> [
            "description"=> "Version of the API used.",

            "type"=> "integer",
            "example"=> 1
        ],
        "customer_support"=> [
            "type" => "object",
            "class" => "CustomerSupport",
            "content"=> "#/components/schemas/customer-support"
        ],
        "hotel_id"=> [
            "description"=> "Echo the item_id (not partner_reference) provided by trivago in the corresponding request.",
            "type"=> "integer"
        ],
        "room_types_array"=> [
            "description"=> "An array of objects detailing the different kinds of rooms that are available for the entire party.\nThe short description must be unique for each room type available in the hotel.",
            "type"=> "array",
            "items"=> [
                "content"=> "#/components/schemas/room-type"
            ]
        ],
//         "hotel_details"=> [
//             "type" => "object",
//             "class" => "HotelDetails",
//             "content"=> "#/components/schemas/hotel-details"
//         ],
        "accepted_credit_cards"=> [
            "description"=> "Accepted Credit Cards",
            "type"=> "array",
            "items"=> [
                "type"=> "string"
            ]
        ],
        "terms_and_conditions"=> [
            "description"=> "General terms and conditions.\nText field.\nLength limit of 1000 characters.",
            "type"=> "string"
        ],
        "terms_and_conditions_url"=> [
            "description"=> "Link to the advertiser's terms and conditions page.",
            "type"=> "string"
        ],
        "payment_policy"=> [
            "description"=> "Describes how the advertiser will use the credit card information, e.g. charged immediately or hold.\nText field.\nLength limit of 1000 characters.",
            "type"=> "string"
        ],
        "other_policy"=> [
            "description"=> "Miscellaneous policies\nText field.\nLength limit of 1000 characters.",
            "type"=> "string"
        ],
        "errors"=> [
            "description"=> "Array of errors",
            "type"=> "array",
            "items"=> [
                "content"=> "#/components/schemas/error"
            ]
        ],
        "start_date"=> [
            "description"=> "The user's check-in date in YYYY-MM-DD",
            "type"=> "string",
            "format"=> "date",
            "example"=> "2019-01-01"
        ],
        "end_date"=> [
            "description"=> "The user's check-out date in YYYY-MM-DD",
            "type"=> "string",
            "format"=> "date",
            "example"=> "2019-01-02"
        ],
        "party"=> [
            "description"=> "Array of JSON objects inlcuding number of adults and children requested.\nEach party object represents the occupation for a separate room.\nFor example, \"party\"=>[[\"adults\"=> 3]] is a request for one room that accommodates three adults,\nor \"party\"=>[[\"adults\"=> 2],[\"adults\"=> 2, \"children\"=> [11,8]]] is a request for one room for two adults\nand one room for two adults and two children aged 11 and 8.",
            "type"=> "array",
            "items"=> [
                "content"=> "#/components/schemas/party"
            ],
            "minItems"=> 1
        ],
        "lang"=> [
            "description"=> "Language code (see Supported Languages)[http=>//developer.trivago.com/expressbooking/supported-languages.html], which identifies the corresponding language and POS.",
            "type"=> "string",
            "example"=> "en_US"
        ],
//         "currency"=> [
//             "description"=> "ISO 4217 currency code. All prices, taxes, fees, and discount amounts should be returned in this requested currency.",
//             "type"=> "string",
//             "example"=> "USD"
//         ],
        "user_country"=> [
            "description"=> "ISO 3166-1 alpha-2 country code corresponding to the user making this request.",
            "type"=> "string",
            "example"=> "US"
        ]
    ];

    public function __construct($request=''){
        parent::__construct(['props'=>$this->_properties]);
        if (!empty($request)){
            foreach ($this->properties as $p => $v) {
                if (isset($request->$p) && $request->$p != ""){
                    $this->properties->$p = $request->$p;
                }
            }
        }
    }
}

class BookingCancelRequest extends Request{
    private $description = "Schema of booking cancel request object";
    protected $required = [
        "partner_reference",
        "reservation_id"
    ];
    private $_properties = [
        "reservation_id" => [
            "description" => "The advertiser's unique id for this reservation. Must be used as booking_id in the tracking pixel.",
            "type" => "string",
            "example" => "XXX123"
        ],
        "partner_reference" => [
            "description" => "The advertiser's unique identifier for this hotel. In a \"/booking_submit\" request this should be the same as\nin the request.",
            "type" => "string",
            "example" => "1"
        ]
    ];

    public function __construct($request=''){
        parent::__construct(['props'=>$this->_properties]);
        $request = preg_replace("/\r|\n|\s/", "", $request);
        $this->origin = $request;
        $request = json_decode($request);
        foreach ($request as $prop_name => $value) {
            if (isset($this->properties->$prop_name)){
                $this->properties->$prop_name = $value;
            }
        }
        //if validate
        $this->_set_adapted_hotel_properties();
    }

    private function _set_adapted_hotel_properties() {
        $this->properties->hms_id = $this->get_hotel_id_from_reference_code($this->properties->partner_reference);
    }

    private function get_hotel_id_from_reference_code($reference_code) {
        $parsed_ids_arr = explode('_', $reference_code);
        return $parsed_ids_arr[0];
    }
}

class BookingCancelResponse extends Response{
    private $title = "Booking Cancel";
    private $description = "Schema of a booking cancellation.";
    protected $required = [
        "partner_reference",
        "reservation_id",
        "status",
        "cancellation_number",
        "customer_support"
    ];
    private $_properties = [
        "customer_support" => [
            "type" => "object",
            "class" => "CustomerSupport",
            "content" => "#/components/schemas/customer-support"
        ],
        "reservation_id" => [
            "description" => "The advertiser's unique id for this reservation. Must be used as booking_id in the tracking pixel.",
            "type" => "string",
            "example" => "XXX123"
        ],
        "partner_reference" => [
            "description" => "The advertiser's unique identifier for this hotel. In a \"/booking_submit\" request this should be the same as\nin the request.",
            "type" => "string",
            "example" => "1"
        ],
        "status" => [
            "enum" => [
                "Success",
                "AlreadyCancelled",
                "UnknownReference",
                "CannotBeCancelled",
                "Error"
            ],
            "type" => "string",
            "example" => "Success"
        ],
        "cancellation_number" => [
            "description" => "Unique identifier for the cancellation. Required if status is \"Success\" or \"AlreadyCancelled.\"",
            "type" => "string",
            "example" => "12345678"
        ]
    ];

    public function __construct($request=''){
        parent::__construct(['props'=>$this->_properties]);
        if (!empty($request)){
            foreach ($this->properties as $p => $v) {
                if (isset($request->$p) && $request->$p != ""){
                    $this->properties->$p = $request->$p;
                }
            }
        }
    }
}

class BookingPaymentAuthoriseRequest{
    private $title = "Booking Payment Confirm Request";
    private $description = "Schema of an booking payment confirm request.";
    private $required = [
        "api_version",
        "reservation_id",
        "callback_parameters",
        "partner_data"
    ];

    private $_properties = [
        "api_version" => [
            "description" => "Version of the API used.",
            "type" => "integer",
            "example" => 1
        ],
        "reservation_id" => [
            "description" => "The advertiser's unique id for the reservation.",
            "type" => "string",
            "example" => "1234"
        ],
        "callback_parameters" => [
            "description" => "Params added by the PSP to the callback URL.",
            "type" => "string",
            "example" => "a=1&b=2"
        ],
        "partner_data" => [
            "content" => "#/components/schemas/partner-data"
        ]
    ];
}

class BookingPaymentAuthoriseResponse{
    private $title = "Booking Payment Confirm";
    private $description = "Schema of an booking payment confirm.";
    private $required = [
        "reference_id",
        "status",
        "customer_support"
    ];
    private $_properties = [
        "customer_support" => [
            "content" => "#/components/schemas/customer-support"
        ],
        "status" => [
            "enum" => [
                "Success",
                "Failure",
                "UnknownReference"
            ],
            "example" => "Success"
        ],
        "problems" => [
            "description" => "An array containing issues that occurred\nRequired, if status = \"Failure.\"\nOptional, if status = \"Success\"",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/problem"
            ]
        ],
        "reference_id" => [
            "description" => "Echo the reference_id provided by trivago in the corresponding request. If the request fails\n(timeout or otherwise), this id can be used in a booking verification request to determine if the call\nwas successful.",
            "type" => "string",
            "example" => "692e8acc-d1d9-45dc-b50f-e7b824f3306f"
        ],
        "reservation" => [
            "content" => "#/components/schemas/reservation"
        ]
    ];
}

class BookingPrepareRequest{
    private $title = "Booking Prepare";
    private $description = "Schema of a booking preparation request.";
    private $required = [
        "checkin_date",
        "checkout_date",
        "partner_reference",
        "reference_id",
        "partner_data",
        "party"
    ];
    private $_properties = [
        "api_version" => [
            "description" => "Version of the API used.",
            "type" => "integer",
            "example" => 1
        ],
        "reference_id" => [
            "description" => "trivago identifier for the booking session. Please consider that this id might not be unique and should not be treated as a \"confirmation\" id.",
            "type" => "string",
            "example" => "692e8acc-d1d9-45dc-b50f-e7b824f3306f"
        ],
        "partner_data" => [
            "content" => "#/components/schemas/partner-data"
        ],
        "party" => [
            "description" => "An array of Party",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/party"
            ],
            "minItems" => 1
        ],
        "checkin_date" => [
            "description" => "The user's check-in date in YYYY-MM-DD.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-01"
        ],
        "checkout_date" => [
            "description" => "The user's check-out date in YYYY-MM-DD.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-02"
        ],
        "partner_reference" => [
            "description" => "Advertiser's unique hotel id.",
            "type" => "string",
            "example" => "1"
        ]
    ];
}

class BookingPrepareResponse{
    private $title = "Booking Prepare Available";
    private $description = "Schema of a booking preparation. oneOf [booking-prepare-response-available || booking-prepare-response-sold-out]";
    private $required = [
        "status",
        "checkin_date",
        "checkout_date",
        "partner_reference",
        "room_type",
        "customer_support"
    ];
    private $_properties = [
        "customer_support" => [
            "content" => "#/components/schemas/customer-support"
        ],
        "status" => [
            "enum" => [
                "ADJUSTED",
                "AVAILABLE",
                "SOLD_OUT"
            ],
            "example" => "AVAILABLE"
        ],
        "room_type" => [
            "content" => "#/components/schemas/room-type"
        ],
        "checkin_date" => [
            "description" => "The user's check-in date in YYYY-MM-DD.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-01"
        ],
        "checkout_date" => [
            "description" => "The user's check-out date in YYYY-MM-DD.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-02"
        ],
        "partner_reference" => [
            "description" => "Advertiser's unique hotel id.",
            "type" => "string",
            "example" => "1"
        ]
    ];
}

class BookingPrepareSoldOut
{
    private $title = "Booking Prepare Sold-Out";
    private $description = "Schema of a booking preparation.";
    private $required = [
        "checkin_date",
        "checkout_date",
        "partner_reference",
        "status",
        "customer_support"
    ];
    private $_properties = [
        "status" => [
            "description" => "Specifies the status of the Prepare response=>\n- SOLD_OUT",
            "type" => "string",
            "enum" => [
                "SOLD_OUT"
            ],
            "example" => "SOLD_OUT"
        ],
        "customer_support" => [
            "content" => "#/components/schemas/customer-support"
        ],
        "checkin_date" => [
            "description" => "The user's check-in date in YYYY-MM-DD.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-01"
        ],
        "checkout_date" => [
            "description" => "The user's check-out date in YYYY-MM-DD.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-02"
        ],
        "partner_reference" => [
            "description" => "Advertiser's unique hotel id.",
            "type" => "string",
            "example" => "1"
        ]
    ];
}

class BookingSubmitRequest extends Request{
    private $title = "Booking Submit Request";
    private $description = "Schema of an booking submit request.";
    private $required = [
        "checkin_date",
        "checkout_date",
        "partner_reference",
        "customer",
        "rooms",
        "final_price_at_booking",
        "final_price_at_checkout",
        "partner_data"
    ];
    private $_properties = [
//         "api_version" => [
//             "description" => "Version of the API used.",
//             "type" => "integer",
//             "example" => 1
//         ],
        "checkin_date" => [
            "description" => "The user's check-in date in YYYY-MM-DD.\nThe check-in date of the traveler.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-01"
        ],
        "checkout_date" => [
            "description" => "The user's check-out date in YYYY-MM-DD.\nThe check-out date of the traveler.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-02"
        ],
        "partner_reference" => [
            "description" => "Advertiser's unique hotel id.\nPartner hotel id.",
            "type" => "string",
            "example" => "1"
        ],
        "ip_address" => [
            "description" => "Sending IP address.",
            "type" => "string"
        ],
        "reference_id" => [
            "description" => "trivago identifier for the booking session. Please consider that this id might not be unique and should not be treated as a \"confirmation\" id. Identifier for the booking session",
            "type" => "string",
            "example" => "692e8acc-d1d9-45dc-b50f-e7b824f3306f"
        ],
        "customer" => [
            "type" => "object",
            "class" => "Customer"
        ],
        "rooms" => [
            "description" => "An array of rooms to be booked.",
            "type" => "array",
            "items" => [
                "type" => "object",
                "class" => "RoomStay",
                "content" => "#/components/schemas/room-stay"
            ],
            "minItems" => 1
        ],
//         "payment_method" => [
//             "description" => "An object containing the credit card details for payment of the reservation.",

//             "content" => "#/components/schemas/booking-submit-request-payment-method"
//         ],
        "payment" => [
            "description" => "Defines the selected payment method of the user.",
            "content" => "#/components/schemas/payment",
            "type" => "object",
            "class" => "Payment",
        ],
        "final_price_at_booking" => [
            "description" => "An object containing the entire price to be paid at time of booking, including taxes and fees.\\nThis will be used to validate the line items.",
            "example" => [
                "amount" => 100,
                "currency" => "USD"
            ],
            "content" => "#/components/schemas/price",
            "type" => "array",
        ],
        "final_price_at_checkout" => [
            "description" => "An object containing the entire price to be paid at time of stay, including taxes and fees.\\nThis will be used to validate the line items. If the full price is paid at the time of booking,\\nthis element must still be included, with an amount of 0 or 0.00.",
            "example" => [
                "amount" => 5,
                "currency" => "USD"
            ],
            "content" => "#/components/schemas/price",
            "type" => "array",
        ],
        "partner_data" => [
            "type" => "array",
            "content" => "#/components/schemas/partner-data"
        ],
        "tracking_data" => [
            "description" => "Enables advertisers to receive specific information for tracking purpose. Fields supported here can be static fields as given by advertiser or can be dynamic fields as supported for tracking pixel which can be fired from express landing page. Refer here for tracking parameters supported from trivago side at submit e.g=>\n\"tracking_data\"=> [\n\"actor\"=> \"trivago\",\n\"tracking_id\"=> \"3125354\"\n]",
            "example" => [
                "actor" => "trivago",
                "advLink" => "http=>//advertiser-site.com/hoteladlon/Executive_Brilliant_Doppelzimmer?start_date=2016-12-28&end_date=2016-12-29"
            ],
            "type" => "array",
        ],
//         "user_agent" => [
//             "description" => "Customer User Agent.",
//             "type" => "string",
//             "example" => "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/74.0.3729.169 Safari/537.36"
//         ]
    ];

    public function __construct($request=''){
        parent::__construct(['props'=>$this->_properties]);
        $request = preg_replace("/\r|\n|\s/", "", $request);
        $this->origin = $request;
        $request = json_decode($request);
        foreach ($request as $prop_name => $value) {
            if (isset($this->properties->$prop_name)){
                $this->properties->$prop_name = $value;
            }
        }
        //if validate
        $this->_set_adapted_hotel_properties();
    }

    private function _set_adapted_hotel_properties() {
        $this->properties->rooms_count = count($this->properties->rooms);
        $this->properties->hms_id = $this->get_hotel_id_from_reference_code($this->properties->partner_reference);
        $this->properties->prop_id = $this->get_property_id_from_reference_code($this->properties->partner_reference);
    }

    private function get_hotel_id_from_reference_code($reference_code) {
        $parsed_ids_arr = explode('_', $reference_code);
        return $parsed_ids_arr[0];
    }

    private function get_property_id_from_reference_code($reference_code) {
        $parsed_ids_arr = explode('_', $reference_code);
        return $parsed_ids_arr[1];
    }
}

class BookingSubmitResponse extends Response
{
    private $title = "Booking Submit Response";
    private $description = "Schema of booking submit response.";
    protected $required = [
        "reference_id",
        "status",
        "customer_support"
    ];
    private $_properties = [
        "customer_support" => [
            "type" => "object",
            "class" => "CustomerSupport",
        ],
        "status" => [
            "type" => "string",
            "enum" => [
                "Success",
                "Failure",
                "PaymentPending",
                "UnknownReference"
            ],
            "example" => "Success"
        ],
//         "authorisation" => [
//             "content" => "#/components/schemas/authorisation"
//         ],
        "problems" => [
            "description" => "An array containing issues that occurred\nRequired, if status = \"Failure.\"\nOptional, if status = \"Success\"",
            "type" => "array",
            "items" => [
                "type" => "object",
                "class" => "Problem",
            ]
        ],
        "reference_id" => [
            "description" => "Echo the reference_id provided by trivago in the corresponding request. If the request fails\n(timeout or otherwise), this id can be used in a booking verification request to determine if the call\nwas successful.",
            "type" => "string",
            "example" => "692e8acc-d1d9-45dc-b50f-e7b824f3306f"
        ],
        "reservation" => [
            "type" => "object",
            "class" => "Reservation",
        ]
    ];

    public function __construct($request=''){
        parent::__construct(['props'=>$this->_properties]);
        if (!empty($request)){
            foreach ($this->properties as $p => $v) {
                if (isset($request->$p) && $request->$p != ""){
                    $this->properties->$p = $request->$p;
                }
            }
        }else{
            $this->properties->problems['description'] = 'Empty response from system ...';
        }
    }
}

class BookingVerifyRequest
{
    private $description = "Schema of booking verify request object";
    private $required = [
        "partner_reference",
        "reservation_id",
        "reference_id"
    ];
    private $_properties = [
        "reservation_id" => [
            "description" => "The advertiser's unique id for this reservation. Must be used as booking_id in the tracking pixel.",
            "type" => "string",
            "example" => "XXX123"
        ],
        "partner_reference" => [
            "description" => "The advertiser's unique identifier for this hotel. In a \"/booking_submit\" request this should be the same as\nin the request.",
            "type" => "string",
            "example" => "1"
        ],
        "reference_id" => [
            "description" => "trivago identifier for the booking session. Please consider that this id might not be unique and should not be\ntreated as a \"confirmation\" id.",
            "type" => "string",
            "example" => "692e8acc-d1d9-45dc-b50f-e7b824f3306f"
        ]
    ];
}

class BookingVerifyResponse
{
    private $title = "Booking Verify";
    private $description = "Schema of an booking verify.";
    private $required = [
        "reference_id",
        "status",
        "customer_support"
    ];
    private $_properties = [
        "customer_support" => [
            "content" => "#/components/schemas/customer-support"
        ],
        "status" => [
            "enum" => [
                "Success",
                "Failure",
                "PaymentPending",
                "UnknownReference"
            ],
            "example" => "Success"
        ],
        "problems" => [
            "description" => "An array containing issues that occurred\nRequired, if status = \"Failure.\"\nOptional, if status = \"Success\"",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/problem"
            ]
        ],
        "reference_id" => [
            "description" => "Echo the reference_id provided by trivago in the corresponding request. If the request fails\n(timeout or otherwise), this id can be used in a booking verification request to determine if the call\nwas successful.",
            "type" => "string",
            "example" => "692e8acc-d1d9-45dc-b50f-e7b824f3306f"
        ],
        "reservation" => [
            "content" => "#/components/schemas/reservation"
        ]
    ];
}

class Address
{
    private $title = "Billing Address object";
    private $description = "Schema of an billing address.";
    private $required = [
        "address1",
        "city",
        "country"
    ];
    private $_properties = [
        "address1" => [
            "description" => "Street and number of address",
            "type" => "string",
            "example" => "12 Johnson Junction"
        ],
        "address2" => [
            "description" => "Additional address information",
            "type" => "string"
        ],
        "city" => [
            "description" => "City of addres",
            "type" => "string",
            "example" => "Santa Cruz"
        ],
        "state" => [
            "description" => "State or province\nState is required for all addresses from the United States (US), Australia (AU), Canada (CA), Malaysia (MY),\nPhilippines (PH) and Italy (IT). For addresses from the United Kingdom (UK), New Zealand (NZ), Ireland (IE),\nSingapore (SG), South Africa (ZA), France (FR) and others the state is optional. required for addresses\nfrom United States.",
            "type" => "string",
            "example" => "Maryland"
        ],
        "postal_code" => [
            "description" => "Zip code or postal code\nrequired for addresses from United States",
            "type" => "string",
            "example" => "97897"
        ],
        "country" => [
            "description" => "ISO 3166 country code",
            "type" => "string",
            "example" => "US"
        ]
    ];
}

class Cancellation
{
    private $title = "Cancellation object";
    private $description = "Schema of an cancellation.";
    private $required = [
        "penalties"
    ];
    private $_properties = [
        "policy" => [
            "description" => "Advertiser's text to describe the cancellation policy.",
            "type" => "string"
        ],
        "penalties" => [
            "description" => "Array to describe the costs user has to pay if the rate is being cancelled.",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/cancellation-penalty"
            ]
        ],
        "type" => [
            "description" => "Specifies the penalty users have to pay at time of booking. Must be of one of the following strings =>\n- unknown\n- full_refundable\n- partial_refundable\n- non_refundable",
            "type" => "string",
            "items" => [
                "enum" => [
                    "unknown",
                    "full_refundable",
                    "partial_refundable",
                    "non_refundable"
                ]
            ]
        ]
    ];
}

class CancellationPenalty
{
    private $title = "Cancellation Penalty object";
    private $description = "Schema of an cancellation penalty.";
    private $required = [
        "fee"
    ];
    private $_properties = [
        "from" => [
            "description" => "Date determined from which time stamp this penalty is applied. Must be in ISO-8601 format, e.g.\n2015-05-25T16=>00=>00+0000. If the penalty is applied at the time of booking, this field must not be provided.",
            "type" => "string",
            "format" => "date-time"
        ],
        "fee" => [
            "content" => "#/components/schemas/price"
        ]
    ];
}

class Contact
{
    private $title = "Contact object";
    private $description = "Schema of an contact.";
    private $required = [
        "contact",
        "description"
    ];
    private $_properties = [
        "contact" => [
            "description" => "Contact details (e.g. phone number or e-mail address).",
            "type" => "string",
            "example" => "444444444"
        ],
        "description" => [
            "description" => "Description of this contact.",
            "type" => "string",
            "example" => "Phone"
        ]
    ];
}

class Customer
{
    private $title = "Customer object";
    private $description = "Schema of an customer.";
    private $required = [
        "first_name",
        "last_name",
        "phone_number",
        "email",
        "country"
    ];
    private $_properties = [
        "first_name" => [
            "description" => "First name of the user submitting the booking.",
            "type" => "string",
            "example" => "Christopher"
        ],
        "last_name" => [
            "description" => "Last name of the user submitting the booking.",
            "type" => "string",
            "example" => "Palmer"
        ],
        "phone_number" => [
            "description" => "Phone number of the user submitting the booking.",
            "type" => "string",
            "example" => "9-(590)311-9256"
        ],
        "email" => [
            "description" => "E-mail address of the user submitting the booking.",
            "type" => "string",
            "example" => "Christopher.Palmer@test.trivago.com"
        ],
        "country" => [
            "description" => "ISO 3166 country code of the user submitting the booking.",
            "type" => "string",
            "example" => "US"
        ]
    ];
}

class CustomerSupport extends BaseTvgClass
{
    private $title = "Customer Support object";
    private $description = "Schema of an customer support.";
    private $required = [
        "phone_numbers",
        "emails"
    ];
    private $_properties = [
        "phone_numbers" => [
            "description" => "Array of customer support phone numbers. The first Contact of the phone_numbers array will be display to the user.",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/contact"
            ]
        ],
        "emails" => [
            "description" => "Array of customer support e-mail addresses.",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/contact"
            ]
        ],
        "urls" => [
            "description" => "Array of url redirecting to customer support information.",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/contact"
            ],
            "deprecated" => true
        ]
    ];
    public function __construct($construct_data=[]){
        parent::__construct(['props'=>$this->_properties]);
    }
}

class Discount
{
    private $title = "Discount object";
    private $description = "Schema of a discount object.";
    private $_properties = [
        "percentage" => [
            "description" => "Percentage value of the discount. Mandatory, only if total is not provided.",
            "type" => "number"
        ],
        "total" => [
            "content" => "#/components/schemas/price"
        ]
    ];
    // "anyOf"=> [
    // [
    // private $required = [
    // "percentage"
    // ]
    // ],
    // [
    // private $required = [
    // "total"
    // ]
    // ]
    // ]
}

class Error
{
    private $title = "Error object";
    private $description = "Schema of an error.";
    private $required = [
        "error_code"
    ];
    private $_properties = [
        "error_code" => [
            "description" => "One of the error codes described below =>\n1. Unknown Error\n2. Request cannot be parsed\n3. Invalid Hotel ID (Should be returned if partner_reference is unknown).\n4. Timeout requested (stops requests for the specified time).\n5. Recoverable Error (equivalent to http 503).",
            "type" => "integer"
        ],
        "message" => [
            "description" => "String describing the error (maximum length of 1000 characters).",
            "type" => "string"
        ],
        "timeout" => [
            "description" => "Number of seconds trivago should stop sending requests (for use with Error Code 4).",
            "type" => "integer"
        ],
        "hotel_ids" => [
            "description" => "Array of hotels ids for which this error occurs.",
            "type" => "array",
            "items" => [
                "oneOf" => [
                    [
                        "type" => "integer"
                    ],
                    [
                        "type" => "string"
                    ]
                ]
            ]
        ]
    ];
}

class Hotel extends BaseTvgClass
{
    private $title = "Hotel object";
    private $description = "Schema of an hotel.";
    private $required = [
        "item_id",
        "partner_reference"
    ];
    private $_properties = [
        "item_id" => [
            "description" => "trivago item id",
            "type" => "integer",
            "example" => 1
        ],
        "partner_reference" => [
            "description" => "Advertiser's unique partner reference",
            "type" => "string",
            "example" => "1"
        ]
    ];
    public function __construct($request=''){
        parent::__construct(['props'=>$this->_properties]);
    }
}

class HotelDetails extends BaseTvgClass
{
    private $title = "Hotel details object";
    private $description = "Schema of a hotel details object";
    private $required = [
        "name",
        "address1",
        "city",
        "country"
    ];
    private $_properties = [
        "name" => [
            "description" => "Name of the hotel",
            "type" => "string"
        ],
        "address1" => [
            "description" => "Street and number of the hotel",
            "type" => "string"
        ],
        "address2" => [
            "description" => "Additional address information about the hotel",
            "type" => "string"
        ],
        "city" => [
            "description" => "City of the hotel",
            "type" => "string"
        ],
        "state" => [
            "description" => "State or province",
            "type" => "string"
        ],
        "postal_code" => [
            "description" => "Zip code or postal code",
            "type" => "string"
        ],
        "country" => [
            "description" => "ISO 3166 country code",
            "type" => "string"
        ],
        "latitude" => [
            "description" => "Latitude coordinate",
            "type" => "number"
        ],
        "longitude" => [
            "description" => "Longitude coordinate",
            "type" => "number"
        ],
        "url" => [
            "description" => "URL for general hotel contact",
            "type" => "string"
        ],
        "phone" => [
            "description" => "Phone number of the hotel",
            "type" => "string"
        ],
        "amenities" => [
            "description" => "Amenities that are generally available at the hotel, not specific to a room. Each string is free text which\nshould be in the language indicated by the 'lang' parameter",
            "type" => "array",
            "items" => [
                "type" => "string"
            ]
        ],
        "photos" => [
            "description" => "URLs for photos of the hotel to be displayed during the booking process. If no photos are available, an empty\narray should be returned.",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/photo"
            ]
        ],
        "checkinout_policy" => [
            "description" => "Describes the check-in/check-out policy for the hotel. Should be in the language indicated by the 'lang'\nparameter. Max length 1000 characters.",
            "type" => "string"
        ]
    ];
    private $deprecated = true;
    public function __construct($construct_data=[]){
        parent::__construct(['props'=>$this->_properties]);
    }
}

class LineItem
{
    private $title = "LineItem object";
    private $description = "Schema of a line item\\n\n|rate model|net_rate|vat|service_fee|booking_fee|hotel_fee|city_tax|resort_fee|others|\n
                                |--- |--- |--- |--- |--- |--- |--- |--- |--- |\n|AI|x|x|x|x|x|x|x|x|\n|GROSS|x|x|x|x|||||\n|NET|x||||||||\n";
    private $required = [
        "price",
        "type",
        "paid_at_checkout",
        "description"
    ];
    private $_properties = [
        "price" => [
            "content" => "#/components/schemas/price"
        ],
        "type" => [
            "description" => "Please note that fields rate, tax and fee are deprecated and should not be used",
            "type" => "string",
            "enum" => [
                "net_rate",
                "city_tax",
                "vat",
                "hotel_fee",
                "resort_fee",
                "service_fee",
                "booking_fee",
                "others",
                "rate",
                "tax",
                "fee"
            ]
        ],
        "paid_at_checkout" => [
            "description" => "true if the rate can be post-paid; false if rate has to be pre-paid.",
            "type" => "boolean"
        ],
        "description" => [
            "description" => "A textual description of the charge. This may be displayed to users and should be in the language indicated by the\n'lang' parameter provided in the response.",
            "type" => "string"
        ]
    ];
}

class PartnerData
{
    private $title = "Partner Data object";
    private $description = "Schema of partner data";
    private $oneOf = [
        [
            "type" => "number",
            "example" => 123
        ],
        [
            "type" => "string",
            "example" => "rate-key"
        ],
        [

            "example" => [
                "rate_key" => "53f3"
            ]
        ],
        [
            "type" => "integer",
            "example" => 123
        ]
    ];
}

class Party
{
    private $title = "Party Object";
    private $description = "Schema of party object.";
    private $required = [
        "adults"
    ];
    private $_properties = [
        "adults" => [
            "description" => "The number of adults staying in the room.",
            "type" => "integer",
            "example" => 2
        ],
        "children" => [
            "description" => "The ages of any children staying in the room.",
            "type" => "array",
            "items" => [
                "type" => "integer"
            ],
            "example" => [
                1
            ]
        ]
    ];
}

class Payment
{
    private $title = "Payment Object";
    private $description = "Schema of payment";
    private $required = [
        "method",
        "parameters",
        "billing_address"
    ];
    private $_properties = [
        "method" => [
            "content" => "#/components/schemas/payment-method"
        ],
        "parameters" => [
            "description" => "Contains the parameters corresponding to the selected payment method.",
            "oneOf" => [
                [
                    "content" => "#/components/schemas/payment-card-parameters"
                ],
                [
                    "content" => "#/components/schemas/external-provider-parameters"
                ],
                [
                    "content" => "#/components/schemas/tokenization-parameters"
                ],
                [
                    "content" => "#/components/schemas/bank-account-parameters"
                ],
                [
                    "content" => "#/components/schemas/manual-payment-parameters"
                ]
            ]
        ],
        "billing_address" => [
            "description" => "Billing address of the user submitting the booking.",
            "content" => "#/components/schemas/address"
        ]
    ];
}

class PaymentAuthenticationNoAuthRequiredResponse
{
    private $title = "No Authenticatoin Required response object";
    private $description = "Schema for No Authentication Required authorisation";
    private $_properties = [
        "type" => [
            "type" => "string",
            "enum" => [
                "NoAuthRequired"
            ],
            "example" => "NoAuthRequired"
        ]
    ];
}

class PaymentAuthentication_3ds1
{
    private $title = "3DS1 authentication object";
    private $description = "Schema of 3DS1 specific authorisation";
    private $required = [
        "cavv",
        "xid",
        "threed_authenticated_response",
        "cavv_algorithm",
        "threed_offered_response",
        "eci",
        "acquirer_code"
    ];
    private $_properties = [
        "type" => [
            "type" => "string",
            "enum" => [
                "3DS1"
            ],
            "example" => "3DS1"
        ],
        "cavv" => [
            "description" => "The Cardholder Authentication Verification Value for the 3DS authentication session",
            "type" => "string"
        ],
        "xid" => [
            "description" => "The 3DS transaction ID of the 3DS session",
            "type" => "string"
        ],
        "threed_authenticated_response" => [
            "description" => "The raw 3DS authentication result from the card issuer.",
            "type" => "string"
        ],
        "cavv_algorithm" => [
            "description" => "The algorithm used to generate the 3DS Cardholder Authentication Verification Value.",
            "type" => "string"
        ],
        "threed_offered_response" => [
            "description" => "The raw enrollment result from the 3DS directory services of the card schemes.",
            "type" => "string"
        ],
        "eci" => [
            "description" => "The Electronic Commerce Indicator returned from the schemes for the 3DS payment session.",
            "type" => "string"
        ],
        "authorisation_mid" => [
            "description" => "This is the MID of the acquirer account that you want to authorise the transaction with.",
            "type" => "string"
        ],
        "acquirer_code" => [
            "description" => "The name of the acquirer processing the payment request.",
            "type" => "string"
        ]
    ];
}

class PaymentAuthentication_3ds2
{
    private $title = "3DS2 authentication object";
    private $description = "Schema of 3DS2 specific authorisation";
    private $required = [
        "authentication_value",
        "ds_trans_id",
        "eci",
        "three_ds_server_trans_id",
        "trans_status",
        "message_version"
    ];
    private $_properties = [
        "type" => [
            "type" => "string",
            "enum" => [
                "3DS2"
            ],
            "example" => "3DS2"
        ],
        "authentication_value" => [
            "description" => "The `authenticationValue` value as defined in the 3D Secure 2 specification.",
            "type" => "string"
        ],
        "ds_trans_id" => [
            "description" => "The `dsTransID` value as defined in the 3D Secure 2 specification.",
            "type" => "string"
        ],
        "eci" => [
            "description" => "The `eci` value as defined in the 3D Secure 2 specification.",
            "type" => "string"
        ],
        "three_ds_server_trans_id" => [
            "description" => "The `threeDSServerTransID` value as defined in the 3D Secure 2 specification.",
            "type" => "string"
        ],
        "trans_status" => [
            "description" => "The `transStatus` value as defined in the 3D Secure 2 specification.",
            "type" => "string"
        ],
        "message_version" => [
            "description" => "The `messageVersion` value as defined in the 3D Secure 2 specification.",
            "type" => "string"
        ]
    ];
}

class BankAccountParameters
{
    private $description = "Bank account parameters";
    private $required = [
        "holder_name",
        "iban",
        "type"
    ];
    private $_properties = [
        "type" => [
            "enum" => [
                "BankAccount"
            ],
            "example" => "BankAccount"
        ],
        "holder_name" => [
            "description" => "Name of account holder.",
            "type" => "string",
            "example" => "Christopher Palmer"
        ],
        "iban" => [
            "description" => "IBAN of bank account.",
            "type" => "string",
            "example" => "DE27100777770209299700"
        ]
    ];
}

class Photo
{
    private $title = "Photo Object";
    private $description = "Schema for the photo";
    private $required = [
        "url"
    ];
    private $_properties = [
        "url" => [
            "description" => "URL for the image. Please assure your images are available with https. The minimum width is 280px, while the\nminimum height is 170px.",
            "type" => "string"
        ],
        "thumbnail_url" => [
            "description" => "URL for thumbnail of the image.",
            "type" => "string"
        ],
        "caption" => [
            "description" => "Caption for the photo. Should be in the language indicated by the 'lang' parameter in the request.",
            "type" => "string"
        ],
        "width" => [
            "description" => "Width in pixels of full size image",
            "type" => "integer"
        ],
        "height" => [
            "description" => "Height in pixels of full size image",
            "type" => "integer"
        ]
    ];
}

class Price
{
    private $title = "Price Object";
    private $description = "Schema for the price";
    private $required = [
        "amount",
        "currency"
    ];
    private $_properties = [
        "amount" => [
            "description" => "The value of the price.",
            "type" => "number",
            "example" => 80
        ],
        "currency" => [
            "description" => "ISO 4217 currency code for the price.",
            "type" => "string",
            "example" => "USD"
        ]
    ];
}

class Problem
{
    private $title = "Problem Object";
    private $description = "Schema of an problem";
    private $required = [
        "problem",
        "explanation"
    ];
    private $_properties = [
        "problem" => [
            "description" => "One of the supported problem types=>",
            "type" => "string",
            "enum" => [
                "AgentAttention",
                "AuthorizationDeclinedByAcquirer",
                "AuthorizationDeclinedByIssuer",
                "CreditCardDeclined",
                "CreditCardTypeNotSupported",
                "InvalidAddress",
                "InvalidCardholderName",
                "InvalidCity",
                "InvalidCountry",
                "InvalidCreditCardExpiry",
                "InvalidCreditCardNumber",
                "InvalidCreditCardSecurityCode",
                "InvalidEmail",
                "InvalidHomePhone",
                "InvalidPostalCode",
                "InvalidReservationFirstName",
                "InvalidReservationLastName",
                "InvalidStateProvince",
                "InvalidTimeRange",
                "InvalidTravelerFirstName",
                "InvalidTravelerLastName",
                "InvalidWorkPhone",
                "MissingAddress",
                "MissingCardholderName",
                "MissingCity",
                "MissingCountry",
                "MissingEmail",
                "MissingHomePhone",
                "MissingPostalCode",
                "MissingReservationFirstName",
                "MissingReservationLastName",
                "MissingStateProvince",
                "MissingTravelerFirstName",
                "MissingTravelerLastName",
                "MissingWorkPhone",
                "MultipleBookingAttempts",
                "PartnerDown",
                "PartnerTimeout",
                "PaymentInsufficientFunds",
                "PaymentRejectedByFraudDetection",
                "PaymentSessionExpired",
                "PendingSupplier",
                "PriceMismatch",
                "PropertyNotSupported",
                "RoomNotAvailable",
                "UnknownPartnerProblem",
                "UnknownReference",
                "UnknownUserProblem",
                "UnsupportedCharSet",
                "UnsupportedOccupancy"
            ]
        ],
        "explanation" => [
            "description" => "Advertiser-specific message to be displayed to the user providing details of the problem, and should be in the\nlanguage of the 'lang' parameter of the request.",
            "type" => "string"
        ],
        "detail" => [
            "description" => "Advertiser-specific message providing details of the problem that will not be displayed to the user. Intended\nfor logging purposes.",
            "type" => "string"
        ]
    ];
}

class Promotion
{
    private $title = "Promotion object";
    private $description = "Schema of a promotion object";
    private $required = [
        "type",
        "description"
    ];
    private $_properties = [
        "type" => [
            "description" => "Type of the promotion. A string value must be one of=> MOBILE_TYPE",
            "type" => "string",
            "enum" => [
                "MOBILE_ONLY"
            ]
        ],
        "description" => [
            "description" => "A textual description of the promotion. This may be displayed to users and should be in the language indicated\nby the 'lang' parameter provided in the response.",
            "type" => "string"
        ],
        "discount" => [
            "content" => "#/components/schemas/discount"
        ]
    ];
}

class Receipt
{
    private $title = "Receipt object";
    private $description = "Schema of an receipt.";
    private $required = [
        "line_items",
        "final_price_at_booking",
        "final_price_at_checkout"
    ];
    private $_properties = [
        "line_items" => [
            "description" => "An array of LineItem objects containing a separate LineItem for each of rate, tax, and fee types.",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/line-item"
            ]
        ],
        "final_price_at_booking" => [
            "content" => "#/components/schemas/price"
        ],
        "final_price_at_checkout" => [
            "content" => "#/components/schemas/price"
        ],
        "profit_margin" => [
            "content" => "#/components/schemas/price"
        ]
    ];
}

class Reservation
{
    private $title = "Reservation object";
    private $description = "Schema of an reservation.";
    private $required = [
        "reservation_id",
        "status",
        "confirmation_url",
        "checkin_date",
        "checkout_date",
        "partner_reference",
        "hotel",
        "customer",
        "rooms",
        "receipt"
    ];
    private $_properties = [
        "reservation_id" => [
            "description" => "The advertiser's unique id for this reservation. Must be used as booking_id in the tracking pixel.",
            "type" => "string",
            "example" => "XXX123"
        ],
        "partner_reference" => [
            "description" => "The advertiser's unique identifier for this hotel. In a \"/booking_submit\" request this should be the same as\nin the request.",
            "type" => "string",
            "example" => "1"
        ],
        "status" => [
            "enum" => [
                "Booked",
                "Cancelled",
                "CheckedIn",
                "CheckedOut"
            ],
            "example" => "Booked"
        ],
        "confirmation_url" => [
            "description" => "Deep link to the reservation confirmation page. The confirmation URL is displayed to the user after a booking has\nbeen confirmed.\nSince this URL is displayed to the user, it has to redirect to a page owned by the booked property which\ndisplays any relevant confirmation codes, and confirms the dates, rooms and final price for the booking.\nPlease provide an empty string, if the confirmation page should be hosted by trivago.",
            "type" => "string",
            "example" => "https=>//advertiser.com/confirmation"
        ],
        "checkin_date" => [
            "description" => "The check-in date of the user in YYYY-MM-DD.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-01"
        ],
        "checkout_date" => [
            "description" => "The check-out date of the user in YYYY-MM-DD.",
            "type" => "string",
            "format" => "date",
            "example" => "2019-01-02"
        ],
        "hotel" => [
            "content" => "#/components/schemas/hotel-details"
        ],
        "customer" => [
            "content" => "#/components/schemas/customer"
        ],
        "rooms" => [
            "description" => "An array of RoomStay objects.",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/room-stay"
            ]
        ],
        "receipt" => [
            "description" => "Information about the pricing on the reservation.",
            "content" => "#/components/schemas/receipt"
        ]
    ];
}

class RoomStay
{
    private $title = "RoomStay Object";
    private $description = "Schema of an room stay.";
    private $required = [
        "party",
        "traveler_first_name",
        "traveler_last_name"
    ];
    private $_properties = [
        "party" => [
            "content" => "#/components/schemas/party"
        ],
        "traveler_first_name" => [
            "description" => "First name of the primary user staying in this room.",
            "type" => "string",
            "example" => "Christopher"
        ],
        "traveler_last_name" => [
            "description" => "Last name of the primary user staying in this room.",
            "type" => "string",
            "example" => "Palmer"
        ]
    ];
}

class RoomType
{
    private $title = "RoomType object";
    private $description = "Schema of an room type.";
    private $required = [
        "name",
        "final_price_at_booking",
        "final_price_at_checkout",
        "partner_data",
        "line_items",
        "occupancy"
    ];
    private $_properties = [
        "name" => [
            "description" => "Short description of the room type. This will be displayed to users, and should be in the language indicated by\nthe 'lang' parameter (max length of 120 characters).",
            "type" => "string",
            "maxLength" => 120
        ],
        "rate_key" => [
            "description" => "Unique identifier for this rate. Used to identify a rate across several \"/booking_availability\" requests. E.g.\nif an user stays on the landing page for a longer period of time before clicking on a specific rate another\n\"/booking_availability\" request is submitted to ensure that the rate clicked is still bookable.\nIf no rate_key is provided a rate is identified based on rate attributes and rate components.",
            "type" => "string"
        ],
        "room_type" => [
            "description" => "Must be one of=>\n- STANDARD\n- COMFORT\n- FAMILY\n- DELUXE\n- SUPERIOR\n- EXECUTIVE\n- JUNIOR_SUITE\n- SUITE\n- EXECUTIVE_SUITE\n- STUDIO\n- APARTMENT\n- BUNGALOW\n- VILLA\n- SHARED\n- OTHERS\nIf room_type field is not provided, room_group.name is used. If room_group.name is missing as well, the field \"name\" of the RoomType object is used.",
            "type" => "string",
            "items" => [
                "enum" => [
                    "STANDARD",
                    "COMFORT",
                    "FAMILY",
                    "DELUXE",
                    "SUPERIOR",
                    "EXECUTIVE",
                    "JUNIOR_SUITE",
                    "SUITE",
                    "EXECUTIVE_SUITE",
                    "STUDIO",
                    "APARTMENT",
                    "BUNGALOW",
                    "VILLA",
                    "SHARED",
                    "OTHERS"
                ]
            ]
        ],
        "room_group" => [
            "description" => "An identifier, which groups different rates to a specified room. All elements with the same room_group.code relates to the same real room (physical).",
            "content" => "#/components/schemas/room-group"
        ],
        "occupancy" => [
            "description" => "The occupancy for this room",
            "content" => "#/components/schemas/occupancy"
        ],
        "final_price_at_booking" => [
            "description" => "Contains the entire price to be paid at time of booking, including taxes and fees. This will be used to validate the line items.",
            "content" => "#/components/schemas/price"
        ],
        "final_price_at_checkout" => [
            "description" => "Contains the entire price to be paid at time of stay, including taxes and fees. This will be used to validate the line items. If the full price is paid at the time of booking, this element must still be included, with an amount of 0 or 0.00.",
            "content" => "#/components/schemas/price"
        ],
        "description" => [
            "description" => "Longer room description. Will be displayed to users, and should be in the language indicated by the 'lang' parameter (max length of 1000 characters).",
            "type" => "string",
            "maxLength" => 1000
        ],
        "rooms_available" => [
            "description" => "The number of rooms remaining/available.",
            "type" => "integer"
        ],
        "partner_data" => [
            "description" => "This data will not be interpreted by trivago, but will be provided to the advertiser when a booking submit request is sent. For example this object can be used to store an advertiser 'rate key,’ ‘room key,’ and/or ‘product key’ . It can consist of a nested JSON objects, or a single string or number.",
            "content" => "#/components/schemas/partner-data"
        ],
        "line_items" => [
            "description" => "Contains the detailed breakdown of charges. An array of LineItem objects will contain a separate LineItem for each of rate, tax, and fee types.",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/line-item"
            ]
        ],
        "amenities" => [
            "description" => "Array of Strings (no html tags allowed). Each String describes an amenity and should be in the language indicated\nby the 'lang' parameter provided in the response.",
            "type" => "array",
            "items" => [
                "type" => "string"
            ]
        ],
        "value_adds" => [
            "description" => "A string value must be one of=>\n- FREE_BREAKFAST\n- FREE_HALF_BOARD\n- FREE_FULL_BOARD\n- FREE_ALL_INCLUSIVE\n- FREE_PARKING\n- FREE_INTERNET\nIf no value adds are supported, an empty array has to be provided.",
            "type" => "array",
            "items" => [
                "enum" => [
                    "FREE_BREAKFAST",
                    "FREE_HALF_BOARD",
                    "FREE_FULL_BOARD",
                    "FREE_ALL_INCLUSIVE",
                    "FREE_PARKING",
                    "FREE_INTERNET"
                ]
            ]
        ],
        "photos" => [
            "description" => "An array of photo objects for the room",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/photo"
            ]
        ],
        "paymentOptions" => [
            "description" => "An array of payment options for the room",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/payment-option"
            ]
        ],
        "payment_methods" => [
            "description" => "Specifies the payment method.",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/payment-method"
            ]
        ],
        "cancellation" => [
            "content" => "#/components/schemas/cancellation"
        ],
        "refundable" => [
            "description" => "Must be one of=>\n- full=> Until day of arrival the reservation may be cancelled without any charge to the user.\n- partial=> At time of booking, the reservation may be cancelled but will incur a charge less than the total reservation amount.\n- none=> At time of booking, the reservation may be cancelled but no refund will be provided.  If the customer has not yet been charged, the full amount will be deducted.\n- unknown=> Indicates that the refundable value cannot be identified as one of the other states.",
            "type" => "string",
            "enum" => [
                "full",
                "partial",
                "none",
                "unknown"
            ],
            "nullable" => true,
            "deprecated" => true
        ],
        "cancellation_policy" => [
            "description" => "Text describing the cancellation policy for the room. Should be in the language indicated by the 'lang'\nparameter. Text field.Length limit=> 1000 characters.",
            "type" => "string",
            "maxLength" => 1000,
            "nullable" => true,
            "deprecated" => true
        ],
        "cancellation_deadline" => [
            "description" => "The datetime after which it is not free to cancel a reservation.The format must match YYYY-MM-DDThh=>mm=>ss. Note\nthere is no timezone info because it's localized to the property.\nExample=> \"cancellation_deadline\"=> \"2015-05-25T16=>00=>00\"",
            "type" => "string",
            "nullable" => true,
            "deprecated" => true
        ],
        "occupancy_policy" => [
            "description" => "Occupancy policies for this room e.g. maximum number of adults, minimum ages.Text field.Length\nlimit=> 1000 characters.",
            "type" => "string",
            "maxLength" => 1000
        ],
        "payment_policy" => [
            "description" => "Describes how the advertiser will use the credit card information, e.g. charged immediately or hold.\n(Length limit=> 1000 characters).",
            "type" => "string",
            "maxLength" => 1000
        ],
        "other_policy" => [
            "description" => "Miscellaneous policies.Text field (Length limit=> 1000 characters).",
            "type" => "string",
            "maxLength" => 5000
        ],
        "promotions" => [
            "description" => "Array of promotions",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/promotion"
            ]
        ]
    ];
}

class Occupancy
{
    private $title = "Occupancy Object";
    private $description = "Schema for an occupancy object";
    private $required = [
        "max_persons",
        "standard_persons",
        "max_adults",
        "max_children",
        "max_children_age"
    ];
    private $_properties = [
        "max_persons" => [
            "description" => "The maximum number of persons (adults and children) allowed to stay in the room.",
            "type" => "integer"
        ],
        "standard_persons" => [
            "description" => "The standard occupancy for this room",
            "type" => "integer"
        ],
        "max_adults" => [
            "description" => "The maximum number of adults allowed to stay in the room.",
            "type" => "integer"
        ],
        "max_children" => [
            "description" => "The maximum number of children allowed to stay in the room",
            "type" => "integer"
        ],
        "max_children_age" => [
            "description" => "The maximum age for a person to be considered as child",
            "type" => "integer"
        ]
    ];
}

class RoomGroup
{

    private $title = "RoomGroup Object";

    private $description = "Schema for a Room Group object";

    private $required = [
        "code",
        "name"
    ];

    private $_properties = [
        "code" => [
            "description" => "A unique room indentifier",
            "type" => "string"
        ],
        "name" => [
            "description" => "Name of the real room (without amenity/rate relations).",
            "type" => "string"
        ],
        "description" => [
            "description" => "Description of the real room limited to 1000 characters.",
            "type" => "string"
        ]
    ];
}

class ErrorResponse
{
    private $title = "Error Response";
    private $description = "Schema response is not OK";
    private $required = [
        "errors",
        "customer_support"
    ];
    private $_properties = [
        "errors" => [
            "description" => "Array of errors",
            "type" => "array",
            "items" => [
                "content" => "#/components/schemas/error"
            ]
        ],
        "customer_support" => [
            "description" => "Contains the hotel’s customer support contact information for guests. Must be returned on both\n     *     successes and failures.",
            "content" => "#/components/schemas/customer-support"
        ]
    ];
}

class Parameters
{
    private $reference_id = [
        "name" => "reference_id",
        "in" => "query",
        "required" => true,
        "schema" => [
            "type" => "string",
            "example" => "692e8acc-d1d9-45dc-b50f-e7b824f3306f"
        ]
    ];
    private $reservation_id = [
        "name" => "reservation_id",
        "in" => "query",
        "required" => true,
        "schema" => [
            "type" => "string",
            "example" => "XXX123"
        ]
    ];
    private $partner_reference = [
        "name" => "partner_reference",
        "in" => "query",
        "required" => true,
        "schema" => [
            "type" => "string",
            "example" => "1"
        ]
    ];
}
